<!DOCTYPE html>
<html>
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>Lista Ofertas Responsiva + Offline</title>
<style>
    /* --- ESTILO GERAL (RESET) --- */
    body, html {
        margin: 0; padding: 0;
        width: 100%; height: 100%;
        overflow: hidden; /* Remove barras de rolagem */
        background-color: #f4f4f4;
        font-family: 'Segoe UI', Helvetica, Arial, sans-serif;
    }

    /* --- CABEÇALHO (15% da altura) --- */
    header {
        height: 15vh; /* Altura relativa */
        background: #d32f2f; /* Vermelho Supermercado */
        color: white;
        display: flex; align-items: center;
        padding: 0 3vw; /* Padding lateral relativo */
        box-shadow: 0 1vh 2vh rgba(0,0,0,0.2);
        position: relative; z-index: 10;
    }

    header h1 {
        font-size: 7vh; /* Fonte gigante relativa */
        margin: 0; flex-grow: 1;
        text-transform: uppercase;
        font-weight: 800; letter-spacing: 0.2vw;
    }

    header .logo-area {
        font-size: 2.5vh;
        font-weight: 600;
        background: white; color: #d32f2f;
        padding: 1vh 2vw;
        border-radius: 5vh;
    }

    /* --- ÁREA DA LISTA (85% da altura) --- */
    #lista-conteiner {
        height: 85vh; 
        padding: 2vh 3vw;
        box-sizing: border-box;
        display: flex;
        flex-direction: column; /* Organiza um embaixo do outro */
        gap: 2vh; /* Espaço entre as linhas */
    }

    /* --- LINHA DO PRODUTO (ITEM) --- */
    .item-row {
        background: white;
        /* FLEXIBILIDADE TOTAL: */
        flex: 1; /* Faz a linha esticar para preencher o espaço */
        border-radius: 2vh;
        display: flex; align-items: center;
        padding: 0 2vw;
        box-shadow: 0 1vh 1vh rgba(0,0,0,0.05);
        border-left: 2vw solid #d32f2f; /* Tarja vermelha */
        animation: slideIn 0.6s ease-out;
        overflow: hidden;
    }

    .item-img {
        height: 80%; /* Ocupa 80% da altura da linha */
        width: auto;
        aspect-ratio: 1/1; /* Tenta manter quadrado */
        object-fit: contain;
        margin-right: 2vw;
        background: #fff;
        border-radius: 1vh;
    }

    .item-nome {
        flex-grow: 1;
        font-size: 5vh; /* Fonte grande responsiva */
        color: #333;
        font-weight: 700;
        white-space: nowrap;
        overflow: hidden;
        text-overflow: ellipsis; /* Três pontinhos se for muito grande */
        line-height: 1.2;
    }

    /* CAIXA DE PREÇO */
    .item-preco-box {
        text-align: right;
        min-width: 20vw; /* Garante espaço para o preço */
        display: flex; align-items: center; justify-content: flex-end;
    }

    .currency {
        font-size: 3vh;
        color: #666; font-weight: bold;
        margin-right: 0.5vw; margin-top: -3vh;
    }

    .valor {
        font-size: 9vh; /* Preço GIGANTE responsivo */
        font-weight: 800; color: #d32f2f;
        line-height: 1;
    }

    .unidade {
        font-size: 2.5vh;
        color: #888; font-weight: 600;
        align-self: flex-end; margin-bottom: 1.5vh; margin-left: 0.5vw;
    }

    /* Rodapé com Paginação */
    .footer-info {
        position: absolute; bottom: 1vh; right: 2vw;
        font-size: 2vh; color: #999;
    }

    /* Status Dot */
    #status-dot { position: fixed; bottom: 5px; right: 5px; width: 15px; height: 15px; border-radius: 50%; background: grey; z-index: 9999; border: 2px solid white; }

    @keyframes slideIn {
        from { opacity: 0; transform: translateX(5vw); }
        to { opacity: 1; transform: translateX(0); }
    }

</style>
</head>
<body>

    <header>
        <h1>Ofertas do Dia</h1>
        <div class="logo-area">Confira nossas promoções</div>
    </header>

    <div id="lista-conteiner">
        </div>
    
    <div class="footer-info" id="paginacao-info"></div>
    <div id="status-dot" title="Status da Conexão"></div>

    <script>
        // CONFIGURAÇÕES
        var API_URL = 'https://mgaplay.com.br/ofertas/destaque-mercado.php'; 
        var ITENS_POR_PAGINA = 5; 
        var TEMPO_POR_PAGINA = 10000; 

        var todosProdutos = [];
        var paginaAtual = 0;
        var loopInterval;
        var erroCont = 0;

        // --- FUNÇÕES VISUAIS ---
        function setStatus(cor) { var dot = document.getElementById('status-dot'); if(dot) dot.style.background = cor; }

        // --- SISTEMA DE CACHE (OFFLINE) ---
        function salvarCache(dados) {
            try { localStorage.setItem('mga_lista_cache', JSON.stringify(dados)); } catch (e) {}
        }

        function carregarCache() {
            try {
                var cache = localStorage.getItem('mga_lista_cache');
                if (cache) {
                    var dados = JSON.parse(cache);
                    if (dados.length > 0) {
                        todosProdutos = dados;
                        setStatus('blue'); // Azul = Cache
                        renderizarPagina();
                        return true;
                    }
                }
            } catch (e) {}
            return false;
        }

        // --- BUSCA DADOS ---
        function buscarDados() {
            var xhr = new XMLHttpRequest();
            var url = API_URL + '?t=' + new Date().getTime();
            
            xhr.open("GET", url, true);
            xhr.timeout = 5000; // Timeout 5s

            xhr.onreadystatechange = function() {
                if (xhr.readyState == 4) {
                    if (xhr.status == 200) {
                        try {
                            var data = JSON.parse(xhr.responseText);
                            // Normaliza array
                            if (Array.isArray(data)) { todosProdutos = data; } else { todosProdutos = [data]; }
                            
                            if (todosProdutos.length > 0) {
                                salvarCache(todosProdutos); // Salva Backup
                                setStatus('lime'); // Verde = Online
                                paginaAtual = 0;
                                renderizarPagina();
                            } else {
                                // Se veio vazio, tenta cache
                                if(todosProdutos.length === 0) carregarCache();
                            }
                        } catch (e) {
                            if(todosProdutos.length === 0) carregarCache();
                            setStatus('orange');
                        }
                    } else {
                        // Erro HTTP
                        if(todosProdutos.length === 0) { if(!carregarCache()) setStatus('red'); }
                        else { setStatus('blue'); } // Mantém o que tem
                    }
                }
            };

            xhr.ontimeout = function () {
                if(todosProdutos.length === 0) carregarCache();
                setStatus('blue');
            };

            xhr.send();
        }

        // --- RENDERIZAÇÃO ---
        function renderizarPagina() {
            const conteiner = document.getElementById('lista-conteiner');
            
            // Se lista vazia, tenta buscar
            if (!todosProdutos || todosProdutos.length === 0) { return; }

            // Lógica de Paginação
            const totalPaginas = Math.ceil(todosProdutos.length / ITENS_POR_PAGINA);
            if (paginaAtual >= totalPaginas) paginaAtual = 0; // Loop

            const inicio = paginaAtual * ITENS_POR_PAGINA;
            const fim = inicio + ITENS_POR_PAGINA;
            const itensDaPagina = todosProdutos.slice(inicio, fim);

            // Limpa e desenha
            conteiner.innerHTML = ''; 

            itensDaPagina.forEach(item => {
                const imgUrl = item.imagem && item.imagem.length > 5 ? item.imagem : 'https://cdn-icons-png.flaticon.com/512/263/263142.png';
                
                // HTML REFEITO COM CLASSES CSS
                const htmlItem = `
                    <div class="item-row">
                        <img src="${imgUrl}" class="item-img" onerror="this.src='https://cdn-icons-png.flaticon.com/512/263/263142.png'">
                        <div class="item-nome">${item.produto || item.titulo}</div>
                        <div class="item-preco-box">
                            <span class="currency">R$</span>
                            <span class="valor">${item.preco}</span>
                            <span class="unidade">/${item.unidade || 'un'}</span>
                        </div>
                    </div>
                `;
                conteiner.insertAdjacentHTML('beforeend', htmlItem);
            });

            // Info Rodapé
            document.getElementById('paginacao-info').innerText = `Página ${paginaAtual + 1} de ${totalPaginas}`;
            
            // Prepara próxima
            paginaAtual++;
        }

        // --- INICIALIZAÇÃO ---
        document.addEventListener("DOMContentLoaded", () => {
            carregarCache(); // Tenta carregar cache primeiro (Instantâneo)
            buscarDados();   // Busca atualização na rede
            
            // Loop de troca de página
            loopInterval = setInterval(() => {
                renderizarPagina();
                // A cada ciclo completo (quando volta pra pag 0), tenta atualizar dados
                if (paginaAtual === 1) buscarDados(); 
            }, TEMPO_POR_PAGINA);
        });

    </script>
</body>
</html>