<!DOCTYPE html>
<html>
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>Lista de Ofertas (Fixa 5 Itens)</title>
<style>
    /* --- RESET & GERAL --- */
    * { margin: 0; padding: 0; box-sizing: border-box; }
    
    body, html {
        width: 100%; height: 100%;
        overflow: hidden;
        background-color: #f4f4f4;
        font-family: 'Roboto', 'Arial', sans-serif; /* Fonte segura para Android */
    }

    /* --- CABEÇALHO (15%) --- */
    header {
        height: 15vh;
        background: #d32f2f;
        color: white;
        display: flex; align-items: center;
        padding: 0 3vw;
        box-shadow: 0 1vh 2vh rgba(0,0,0,0.2);
        position: relative; z-index: 10;
    }

    header h1 {
        font-size: 7vh;
        margin: 0; flex-grow: 1;
        text-transform: uppercase;
        font-weight: 800;
        letter-spacing: 0.1vw;
    }

    /* --- ÁREA DA LISTA (85%) --- */
    #lista-conteiner {
        height: 85vh; 
        padding: 2vh 3vw;
        display: flex;
        flex-direction: column;
        gap: 1.5vh; /* Espaço entre as linhas */
    }

    /* --- LINHA DO PRODUTO --- */
    .item-row {
        background: white;
        /* IMPORTANTE: flex: 1 faz com que todas as 5 linhas tenham altura igual */
        flex: 1; 
        border-radius: 2vh;
        display: flex; align-items: center;
        padding: 0 2vw;
        box-shadow: 0 0.5vh 1vh rgba(0,0,0,0.05);
        border-left: 2vw solid #d32f2f;
        animation: slideIn 0.6s ease-out;
        overflow: hidden;
    }

    /* Classe para preencher espaço vazio se tiver menos de 5 produtos */
    .linha-vazia {
        background: transparent !important;
        box-shadow: none !important;
        border: none !important;
        flex: 1; 
    }

    .item-img {
        height: 85%; 
        width: auto;
        aspect-ratio: 1/1; 
        object-fit: contain;
        margin-right: 2vw;
        background: #fff;
        border-radius: 1vh;
        border: 1px solid #eee;
    }

    .item-nome {
        flex-grow: 1;
        font-size: 5vh; 
        color: #333;
        font-weight: 700;
        white-space: nowrap;
        overflow: hidden;
        text-overflow: ellipsis;
        line-height: 1.2;
    }

    /* CAIXA DE PREÇO */
    .item-preco-box {
        text-align: right;
        min-width: 22vw; 
        display: flex; align-items: center; justify-content: flex-end;
    }

    .currency {
        font-size: 3.5vh;
        color: #666; font-weight: bold;
        margin-right: 0.5vw; margin-top: -3vh;
    }

    .valor {
        font-size: 9vh; 
        font-weight: 800; color: #d32f2f;
        line-height: 1;
    }

    .unidade {
        font-size: 2.5vh;
        color: #888; font-weight: 600;
        align-self: flex-end; margin-bottom: 1.5vh; margin-left: 0.5vw;
        text-transform: lowercase;
    }

    /* Rodapé com Paginação */
    .footer-info {
        position: absolute; bottom: 1vh; right: 2vw;
        font-size: 2vh; color: #999;
    }

    /* Status Dot */
    #status-dot { 
        position: fixed; bottom: 5px; right: 5px; 
        width: 15px; height: 15px; border-radius: 50%; 
        background: grey; z-index: 9999; border: 2px solid white; 
    }

    @keyframes slideIn {
        from { opacity: 0; transform: translateX(5vw); }
        to { opacity: 1; transform: translateX(0); }
    }

</style>
</head>
<body>

    <header>
        <h1>Ofertas do Dia</h1>
    </header>

    <div id="lista-conteiner">
        </div>
    
    <div class="footer-info" id="paginacao-info"></div>
    <div id="status-dot" title="Status da Conexão"></div>

    <script>
        // --- CONFIGURAÇÕES ---
        var API_URL = 'https://mgaplay.com.br/ofertas/destaque-mercado.php'; 
        var ITENS_POR_PAGINA = 5; // EXATAMENTE 5 ITENS
        var TEMPO_POR_PAGINA = 10000; // 10 segundos por página

        var todosProdutos = [];
        var paginaAtual = 0;
        var placeholder = 'https://cdn-icons-png.flaticon.com/512/263/263142.png';

        // --- FORMATAR PREÇO (R$ 00,00) ---
        function formatarMoeda(valor) {
            if (!valor) return "0,00";
            // Garante que é string, troca virgula por ponto para converter
            let v = String(valor).replace(',', '.');
            let floatV = parseFloat(v);
            if (isNaN(floatV)) return valor; // Retorna original se der erro
            // Formata com 2 casas e troca ponto por virgula
            return floatV.toFixed(2).replace('.', ',');
        }

        function setStatus(cor) { document.getElementById('status-dot').style.background = cor; }

        // --- CACHE OFFLINE ---
        function salvarCache(dados) {
            try { localStorage.setItem('mga_tvtab_cache', JSON.stringify(dados)); } catch (e) {}
        }

        function carregarCache() {
            try {
                var cache = localStorage.getItem('mga_tvtab_cache');
                if (cache) {
                    var dados = JSON.parse(cache);
                    if (dados.length > 0) {
                        todosProdutos = dados;
                        setStatus('blue'); // Azul = Cache
                        renderizarPagina();
                        return true;
                    }
                }
            } catch (e) {}
            return false;
        }

        // --- BUSCA DADOS ---
        function buscarDados() {
            // Pega UID da URL se existir (?uid=123)
            var urlParams = new URLSearchParams(window.location.search);
            var uid = urlParams.get('uid') || urlParams.get('loja') || 1;

            var xhr = new XMLHttpRequest();
            var url = API_URL + '?uid=' + uid + '&t=' + new Date().getTime();
            
            xhr.open("GET", url, true);
            xhr.timeout = 5000;

            xhr.onreadystatechange = function() {
                if (xhr.readyState == 4) {
                    if (xhr.status == 200) {
                        try {
                            var data = JSON.parse(xhr.responseText);
                            if (Array.isArray(data)) { todosProdutos = data; } else { todosProdutos = [data]; }
                            
                            if (todosProdutos.length > 0) {
                                salvarCache(todosProdutos);
                                setStatus('lime'); // Verde = Online
                                paginaAtual = 0;
                                renderizarPagina();
                            } else {
                                if(todosProdutos.length === 0) carregarCache();
                            }
                        } catch (e) {
                            if(todosProdutos.length === 0) carregarCache();
                        }
                    } else {
                        if(todosProdutos.length === 0) carregarCache();
                    }
                }
            };
            xhr.onerror = function() { if(todosProdutos.length === 0) carregarCache(); };
            xhr.send();
        }

        // --- RENDERIZAÇÃO ---
        function renderizarPagina() {
            const conteiner = document.getElementById('lista-conteiner');
            
            if (!todosProdutos || todosProdutos.length === 0) return;

            // Loop de página
            const totalPaginas = Math.ceil(todosProdutos.length / ITENS_POR_PAGINA);
            if (paginaAtual >= totalPaginas) paginaAtual = 0;

            const inicio = paginaAtual * ITENS_POR_PAGINA;
            const itensDaPagina = todosProdutos.slice(inicio, inicio + ITENS_POR_PAGINA);

            conteiner.innerHTML = ''; 

            // 1. Renderiza os itens que existem
            itensDaPagina.forEach(item => {
                const imgUrl = (item.imagem && item.imagem.length > 5) ? item.imagem : placeholder;
                
                const htmlItem = `
                    <div class="item-row">
                        <img src="${imgUrl}" class="item-img" onerror="this.src='${placeholder}'">
                        <div class="item-nome">${item.produto}</div>
                        <div class="item-preco-box">
                            <span class="currency">R$</span>
                            <span class="valor">${formatarMoeda(item.preco)}</span>
                            <span class="unidade">/${item.unidade || 'un'}</span>
                        </div>
                    </div>
                `;
                conteiner.insertAdjacentHTML('beforeend', htmlItem);
            });

            // 2. Preenche com linhas vazias se não tiver 5 itens (para manter o layout fixo)
            while(conteiner.children.length < ITENS_POR_PAGINA) {
                conteiner.insertAdjacentHTML('beforeend', '<div class="linha-vazia"></div>');
            }

            // Info Rodapé
            document.getElementById('paginacao-info').innerText = `Página ${paginaAtual + 1} de ${totalPaginas}`;
            
            paginaAtual++;
        }

        // --- INICIALIZAÇÃO ---
        window.onload = function() {
            carregarCache(); 
            buscarDados();   
            
            setInterval(function() {
                renderizarPagina();
                // A cada volta completa, tenta atualizar os dados
                if (paginaAtual === 1) buscarDados(); 
            }, TEMPO_POR_PAGINA);
        };

    </script>
</body>
</html>